/*
# PostgreSQL Database Modeler (pgModeler)
#
# Copyright 2006-2019 - Raphael Araújo e Silva <raphael@pgmodeler.io>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation version 3.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# The complete text of GPLv3 is at LICENSE file on source code root directory.
# Also, you can get the complete GNU General Public License at <http://www.gnu.org/licenses/>
*/

#include "operatorclasselement.h"

OperatorClassElement::OperatorClassElement(void)
{
	element_type=OperatorElem;
	function=nullptr;
	_operator=nullptr;
	strategy_number=0;
	op_family=nullptr;
}

void OperatorClassElement::setFunction(Function *func, unsigned stg_number)
{
	//Raises an exception case the function is not allocated
	if(!func)
		throw Exception(ErrorCode::AsgNotAllocattedObject,__PRETTY_FUNCTION__,__FILE__,__LINE__);

	//Raises an error case the stratagy number is zero (invalid)
	else if(stg_number==0)
		throw Exception(ErrorCode::AsgInvalidSupportStrategyNumber,__PRETTY_FUNCTION__,__FILE__,__LINE__);

	//Clear the attributes not related to the FUNCTION element type
	this->_operator=nullptr;
	this->op_family=nullptr;

	//Configure the attributes that belongs to the element type
	this->function=func;
	this->strategy_number=stg_number;
	this->element_type=FunctionElem;
}

void OperatorClassElement::setOperator(Operator *oper, unsigned stg_number)
{
	//Raises an error in case the operator is not allocated
	if(!oper)
		throw Exception(ErrorCode::AsgNotAllocattedObject,__PRETTY_FUNCTION__,__FILE__,__LINE__);

	//Raises an error case the stratagy number is zero (invalid)
	else if(stg_number==0)
		throw Exception(ErrorCode::AsgInvalidSupportStrategyNumber,__PRETTY_FUNCTION__,__FILE__,__LINE__);

	//Clear the attributes not related to the OPERATOR element type
	this->function=nullptr;

	//Configure the attributes that belongs to the element type
	this->_operator=oper;
	this->strategy_number=stg_number;
	this->element_type=OperatorElem;
}

void OperatorClassElement::setOperatorFamily(OperatorFamily *op_family)
{
	if(this->element_type==OperatorElem)
	{
		if(op_family && op_family->getIndexingType()!=IndexingType::Btree)
			throw Exception(ErrorCode::AsgInvalidOpFamilyOpClassElem,__PRETTY_FUNCTION__,__FILE__,__LINE__);

		this->op_family=op_family;
	}
}

void OperatorClassElement::setStorage(PgSqlType storage)
{
	//Clear the attributes not related to the STORAGE element type
	this->function=nullptr;
	this->_operator=nullptr;
	this->op_family=nullptr;
	this->strategy_number=0;

	//Configure the attributes that belongs to the element type
	this->storage=storage;
	this->element_type=StorageElem;
}

unsigned OperatorClassElement::getElementType(void)
{
	return(element_type);
}

Function *OperatorClassElement::getFunction(void)
{
	return(function);
}

Operator *OperatorClassElement::getOperator(void)
{
	return(_operator);
}

OperatorFamily *OperatorClassElement::getOperatorFamily(void)
{
	return(op_family);
}

PgSqlType OperatorClassElement::getStorage(void)
{
	return(storage);
}

unsigned OperatorClassElement::getStrategyNumber(void)
{
	return(strategy_number);
}

QString OperatorClassElement::getCodeDefinition(unsigned def_type)
{
	SchemaParser schparser;
	attribs_map attributes;

	attributes[Attributes::Type]=QString();
	attributes[Attributes::StrategyNum]=QString();
	attributes[Attributes::Signature]=QString();
	attributes[Attributes::Function]=QString();
	attributes[Attributes::Operator]=QString();
	attributes[Attributes::Storage]=QString();
	attributes[Attributes::OpFamily]=QString();
	attributes[Attributes::Definition]=QString();

	if(element_type==FunctionElem && function && strategy_number > 0)
	{
		//FUNCTION support_number [ ( op_type [ , op_type ] ) ] funcname ( argument_type [, ...] )
		attributes[Attributes::Function]=Attributes::True;
		attributes[Attributes::StrategyNum]=QString("%1").arg(strategy_number);

		if(def_type==SchemaParser::SqlDefinition)
			attributes[Attributes::Signature]=function->getSignature();
		else
			attributes[Attributes::Definition]=function->getCodeDefinition(def_type,true);
	}
	else if(element_type==OperatorElem && _operator && strategy_number > 0)
	{
		//OPERATOR strategy_number operator_name [ ( op_type, op_type ) ] [ FOR SEARCH | FOR ORDER BY sort_family_name ]
		attributes[Attributes::Operator]=Attributes::True;
		attributes[Attributes::StrategyNum]=QString("%1").arg(strategy_number);

		if(def_type==SchemaParser::SqlDefinition)
			attributes[Attributes::Signature]=_operator->getSignature();
		else
			attributes[Attributes::Definition]=_operator->getCodeDefinition(def_type,true);

		if(op_family)
		{
			if(def_type==SchemaParser::SqlDefinition)
				attributes[Attributes::OpFamily]=op_family->getName(true);
			else
				attributes[Attributes::Definition]+=op_family->getCodeDefinition(def_type,true);
		}
	}
	else if(element_type==StorageElem && storage!=PgSqlType::Null)
	{
		//STORAGE storage_type
		attributes[Attributes::Storage]=Attributes::True;

		if(def_type==SchemaParser::SqlDefinition)
			attributes[Attributes::Type]=(*storage);
		else
			attributes[Attributes::Definition]=storage.getCodeDefinition(def_type);
	}

	return(schparser.getCodeDefinition(Attributes::Element,attributes, def_type));
}

bool OperatorClassElement::operator == (OperatorClassElement &elem)
{
	return(this->element_type == elem.element_type &&
		   this->storage == elem.storage &&
		   this->function == elem.function &&
		   this->_operator == elem._operator &&
		   this->strategy_number == elem.strategy_number &&
		   this->op_family == elem.op_family);
}

