//
// File: test_timsframe.cpp
// Created by: Olivier Langella
// Created on: 12/4/2020
//
/*******************************************************************************
 * Copyright (c) 2020 Olivier Langella <olivier.langella@u-psud.fr>.
 *
 * This file is part of the PAPPSOms++ library.
 *
 *     PAPPSOms++ is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms++ is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms++.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/

// make test ARGS="-V -I 1,1"

// ./tests/catch2-only-tests [timsframe] -s
// ./tests/catch2-only-tests [cubicsolver] -s


// ./tests/catch2-only-tests [timsframe] -s -a

#include <catch2/catch_test_macros.hpp>
#include <catch2/catch_approx.hpp>
#include <catch2/matchers/catch_matchers_vector.hpp>

#include <QCoreApplication>
#include <QDebug>
#include <QString>
#include <iostream>
#include <pappsomspp/core/vendors/tims/timsframebase.h>
#include <pappsomspp/core/vendors/tims/mzcalibration/cardano.h>
#include <pappsomspp/core/mzrange.h>
#include <pappsomspp/core/msfile/msfileaccessor.h>
#include <pappsomspp/core/msrun/private/timsmsrunreaderms2.h>
#include <pappsomspp/core/msrun/private/timsmsrunreader.h>
#include <pappsomspp/core/pappsoexception.h>
#include <pappsomspp/core/exception/exceptionnotfound.h>
#include <pappsomspp/core/processing/filters/filtersuitestring.h>
#include <pappsomspp/core/processing/filters/filterresample.h>
#include "tests/tests-config.h"

using namespace pappso;
using namespace std;


TEST_CASE("Test Cardano's cubic solver", "[cubicsolver]")
{

  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  INFO("..:: Test Cardano's cubic solver begin ::..");
  //   OboPsiMod test;


  SECTION("Test cubic")
  {


    std::vector<double> X;
    X.push_back(18);
    X.push_back(56);
    X.push_back(3);
    X.push_back(2);


    // REQUIRE(m[0].x == Approx(-0.32588));

    InHousePolynomialSolverResult res;
    res = inHousePolynomialSolve(X);
    REQUIRE(res.type == CardanoResultCase::positived);
    // https://www.wolframalpha.com/widgets/view.jsp?id=578d50248844454e46e24e9ed230843d

    REQUIRE(res.x1 == Catch::Approx(-0.32588));
  }


  SECTION("Test quadratic")
  {


    std::vector<double> X;
    X.push_back(18);
    X.push_back(56);
    X.push_back(3);
    // X.push_back(2);


    // REQUIRE(m[0].x == Approx(-0.3271626093));

    InHousePolynomialSolverResult res;
    res = inHousePolynomialSolve(X);
    REQUIRE(res.type == CardanoResultCase::quadratic);
    // https://www.wolframalpha.com/widgets/view.jsp?id=578d50248844454e46e24e9ed230843d

    REQUIRE(res.x1 == Catch::Approx(-0.3271626093));
  }


  SECTION("Test line")
  {


    std::vector<double> X;
    X.push_back(18);
    X.push_back(56);
    // X.push_back(3);
    // X.push_back(2);


    // REQUIRE(m[0].x == Approx(-0.3214285714));pappsomspp

    InHousePolynomialSolverResult res;
    res = inHousePolynomialSolve(X);
    REQUIRE(res.type == CardanoResultCase::line);
    // https://www.wolframalpha.com/widgets/view.jsp?id=578d50248844454e46e24e9ed230843d

    REQUIRE(res.x1 == Catch::Approx(-0.3214285714));
  }
}

TEST_CASE("Test tims frames", "[timsframe]")
{

  // Set the debugging message formatting pattern.
  qSetMessagePattern(QString("%{file}@%{line}, %{function}(): %{message}"));

  INFO("..:: Test TIMS frame init begin ::..");
  //   OboPsiMod test;

  TimsFrameBase frame(1, 671);

  MzRange mz_range(1200.0001, PrecisionFactory::getPpmInstance(10));

  SECTION("Test 1")
  {
    double temperature_correction =
      77.0 * (25.4933665127396 - 25.59978165276) + (-3.7) * (26.2222755503081 - 27.6311281556638);
    temperature_correction = (double)1.0 + (temperature_correction / 1.0e6);


    frame.setMzCalibration(25.59978165276,
                           27.6311281556638,
                           0.2,
                           24864.0,
                           313.577620892277,
                           157424.07710945,
                           0.000338743021989553,
                           0.0,
                           0.0,

                           25.4933665127396,
                           26.2222755503081,
                           77.0,
                           -3.7);
    frame.setTimsCalibration(2,
                             1,
                             670,
                             207.775676931964,
                             59.2526676368822,
                             33.0,
                             1,
                             0.0209889001473149,
                             131.440113097798,
                             12.9712317295887,
                             2558.71692505931);
    frame.setRtInSeconds(2402.64305686123);
    frame.setMsMsType(0);
    quint32 index =
      frame.getMzCalibrationInterfaceSPtr().get()->getTofIndexFromMz(mz_range.getMz());
    qDebug() << "index=" << index;
    double mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(index);
    qDebug() << "mz=" << QString::number(mz, 'g', 10);

    REQUIRE(mz_range.contains(mz));

    // 313792
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex((quint32)313793);

    REQUIRE(mz == Catch::Approx(1200.002299));
    qDebug() << " 313793 => mz=" << QString::number(mz, 'g', 10);
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex((quint32)313792);

    REQUIRE(mz == Catch::Approx(1200.002299));
    qDebug() << " 313792 => mz=" << QString::number(mz, 'g', 10);
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex((quint32)313791);
    qDebug() << " 313791 => mz=" << QString::number(mz, 'g', 10);


    double one_over_k0 = frame.getOneOverK0Transformation(400);
    qDebug() << " 400 => one_over_k0=" << QString::number(one_over_k0, 'g', 10);
    std::size_t scan_num = frame.getScanIndexFromOneOverK0(one_over_k0);

    REQUIRE(scan_num == 400);
    std::cout << std::endl << "..:: Test 1 OK ::.." << std::endl;
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex((quint32)500000);
    REQUIRE(mz == Catch::Approx(2442.0497183326));
  }

  SECTION("Test 2")
  {
    std::cout << std::endl << "..:: second test, new calibration parameters ::.." << std::endl;


    frame.setMzCalibration(25.3072302808429,
                           25.3072302808429,
                           0.2,
                           24864.0,
                           313.577620892277,
                           157424.07710945,
                           0.000338743021989553,
                           0.0,
                           0.0,

                           25.4933665127396,
                           26.2222755503081,
                           77.0,
                           -3.7);
    /*
          q.value(2).toDouble(),  // MzCalibration.DigitizerTimebase
          q.value(3).toDouble(),  // MzCalibration.DigitizerDelay
          q.value(4).toDouble(),  // MzCalibration.C0
          q.value(5).toDouble(),  // MzCalibration.C1
          q.value(6).toDouble(),  // MzCalibration.C2
          q.value(7).toDouble()); // MzCalibration.C3
          */

    frame.setTimsCalibration(2,
                             1,
                             1537,
                             179.152532325778,
                             90.4208212951646,
                             33.0,
                             1,
                             0.009221,
                             131.053614,
                             9.656516,
                             2095.24199);
    frame.setRtInSeconds(1.7950832);
    frame.setMsMsType(0);

    std::size_t index =
      frame.getMzCalibrationInterfaceSPtr().get()->getTofIndexFromMz(mz_range.getMz());
    qDebug() << "index=" << index;
    double mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex(index);
    qDebug() << "mz=" << QString::number(mz, 'g', 10);

    REQUIRE(mz_range.contains(mz));

    // 313792
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex((quint32)313793);
    qDebug() << " 313793 => mz=" << QString::number(mz, 'g', 10);
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex((quint32)313792);
    qDebug() << " 313792 => mz=" << QString::number(mz, 'g', 10);
    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex((quint32)313791);
    qDebug() << " 313791 => mz=" << QString::number(mz, 'g', 10);


    double one_over_k0 = frame.getOneOverK0Transformation(400);
    qDebug() << " 400 => one_over_k0=" << QString::number(one_over_k0, 'g', 10);
    std::size_t scan_num = frame.getScanIndexFromOneOverK0(one_over_k0);

    REQUIRE(scan_num == 400);


    mz = frame.getMzCalibrationInterfaceSPtr().get()->getMzFromTofIndex((quint32)375176);
    qDebug() << " 375176 => mz=" << QString::number(mz, 'g', 10);


    REQUIRE_FALSE(mz < 50.0);

    REQUIRE(frame.getMzCalibrationInterfaceSPtr().get()->getTofIndexFromMz(1531.6385523357) ==
            370436);
    REQUIRE(frame.getMzCalibrationInterfaceSPtr().get()->getTofIndexFromMz(1531.6561258431) ==
            370439);

    std::cout << std::endl << "..:: Test 2 OK ::.." << std::endl;
  }
}
