// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
/**
 * @class   vtkImplicitDataSet
 * @brief   treat a dataset as if it were an implicit function
 *
 * vtkImplicitDataSet treats any type of dataset as if it were an
 * implicit function. This means it computes a function value and
 * gradient. vtkImplicitDataSet is a concrete implementation of
 * vtkImplicitFunction.
 *
 * vtkImplicitDataSet computes the function (at the point x) by performing
 * cell interpolation. That is, it finds the cell containing x, and then
 * uses the cell's interpolation functions to compute an interpolated
 * scalar value at x. (A similar approach is used to find the
 * gradient, if requested.) Points outside of the dataset are assigned
 * the value of the ivar OutValue, and the gradient value OutGradient.
 *
 * @warning
 * Any type of dataset can be used as an implicit function as long as it
 * has scalar data associated with it.
 *
 * @sa
 * vtkImplicitFunction vtkImplicitVolume vtkClipPolyData vtkCutter
 * vtkImplicitWindowFunction
 */

#ifndef vtkImplicitDataSet_h
#define vtkImplicitDataSet_h

#include "vtkCommonDataModelModule.h" // For export macro
#include "vtkImplicitFunction.h"

VTK_ABI_NAMESPACE_BEGIN
class vtkDataSet;

class VTKCOMMONDATAMODEL_EXPORT vtkImplicitDataSet : public vtkImplicitFunction
{
public:
  vtkTypeMacro(vtkImplicitDataSet, vtkImplicitFunction);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   * Construct an vtkImplicitDataSet with no initial dataset; the OutValue
   * set to a large negative number; and the OutGradient set to (0,0,1).
   */
  static vtkImplicitDataSet* New();

  /**
   * Return the MTime also considering the DataSet dependency.
   */
  vtkMTimeType GetMTime() override;

  ///@{
  /**
   * Evaluate the implicit function. This returns the interpolated scalar value
   * at x[3].
   */
  using vtkImplicitFunction::EvaluateFunction;
  double EvaluateFunction(double x[3]) override;
  ///@}

  /**
   * Evaluate implicit function gradient.
   */
  void EvaluateGradient(double x[3], double n[3]) override;

  ///@{
  /**
   * Set / get the dataset used for the implicit function evaluation.
   */
  virtual void SetDataSet(vtkDataSet*);
  vtkGetObjectMacro(DataSet, vtkDataSet);
  ///@}

  ///@{
  /**
   * Set / get the function value to use for points outside of the dataset.
   */
  vtkSetMacro(OutValue, double);
  vtkGetMacro(OutValue, double);
  ///@}

  ///@{
  /**
   * Set / get the function gradient to use for points outside of the dataset.
   */
  vtkSetVector3Macro(OutGradient, double);
  vtkGetVector3Macro(OutGradient, double);
  ///@}

protected:
  vtkImplicitDataSet();
  ~vtkImplicitDataSet() override;

  void ReportReferences(vtkGarbageCollector*) override;

  vtkDataSet* DataSet;
  double OutValue;
  double OutGradient[3];

  double* Weights; // used to compute interpolation weights
  int Size;        // keeps track of length of weights array

private:
  vtkImplicitDataSet(const vtkImplicitDataSet&) = delete;
  void operator=(const vtkImplicitDataSet&) = delete;
};

VTK_ABI_NAMESPACE_END
#endif
