// Copyright 2021 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import 'chrome://resources/cr_elements/icons.m.js';
import 'chrome://resources/cr_elements/shared_vars_css.m.js';

import {html, PolymerElement} from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';

import {AddSinkResultCode} from '../access_code_cast.mojom-webui.js';
import {RouteRequestResultCode} from '../route_request_result_code.mojom-webui.js';

enum ErrorMessage {
  NO_ERROR,
  GENERIC,
  ACCESS_CODE,
  NETWORK,
  PERMISSION,
  TOO_MANY_REQUESTS,
}

export class ErrorMessageElement extends PolymerElement {
  private static readonly ADD_RESULT_MESSAGE_CODES:
      [ErrorMessage, AddSinkResultCode[]][] = [
        [ErrorMessage.NO_ERROR, [AddSinkResultCode.OK]],
        [
          ErrorMessage.GENERIC,
          [
            AddSinkResultCode.UNKNOWN_ERROR,
            AddSinkResultCode.SINK_CREATION_ERROR,
            AddSinkResultCode.CHANNEL_OPEN_ERROR,
            // TODO(b/216529759): Make a new ErrorMessage for profile sync
            // error.
            AddSinkResultCode.PROFILE_SYNC_ERROR
          ]
        ],
        [
          ErrorMessage.ACCESS_CODE,
          [
            AddSinkResultCode.INVALID_ACCESS_CODE,
            AddSinkResultCode.ACCESS_CODE_NOT_FOUND
          ]
        ],
        [
          ErrorMessage.NETWORK,
          [
            AddSinkResultCode.HTTP_RESPONSE_CODE_ERROR,
            AddSinkResultCode.RESPONSE_MALFORMED,
            AddSinkResultCode.EMPTY_RESPONSE,
            AddSinkResultCode.SERVICE_NOT_PRESENT,
            AddSinkResultCode.SERVER_ERROR
          ]
        ],
        [ErrorMessage.PERMISSION, [AddSinkResultCode.AUTH_ERROR]],
        [ErrorMessage.TOO_MANY_REQUESTS, [AddSinkResultCode.TOO_MANY_REQUESTS]],
      ];

  private static readonly CAST_RESULT_MESSAGE_CODES:
      [ErrorMessage, RouteRequestResultCode[]][] = [
        [ErrorMessage.NO_ERROR, [RouteRequestResultCode.OK]],
        [
          ErrorMessage.GENERIC,
          [
            RouteRequestResultCode.UNKNOWN_ERROR,
            RouteRequestResultCode.INVALID_ORIGIN,
            RouteRequestResultCode.DEPRECATED_OFF_THE_RECORD_MISMATCH,
            RouteRequestResultCode.NO_SUPPORTED_PROVIDER,
            RouteRequestResultCode.CANCELLED,
            RouteRequestResultCode.ROUTE_ALREADY_EXISTS,
            RouteRequestResultCode.DESKTOP_PICKER_FAILED,
            RouteRequestResultCode.ROUTE_ALREADY_TERMINATED
          ]
        ],
        [
          ErrorMessage.NETWORK,
          [
            RouteRequestResultCode.TIMED_OUT,
            RouteRequestResultCode.ROUTE_NOT_FOUND,
            RouteRequestResultCode.SINK_NOT_FOUND
          ]
        ],
      ];

  private static readonly ADD_RESULT_MESSAGE_MAP =
      new Map(ErrorMessageElement.ADD_RESULT_MESSAGE_CODES);

  private static readonly CAST_RESULT_MESSAGE_MAP =
      new Map(ErrorMessageElement.CAST_RESULT_MESSAGE_CODES);

  // Needed for Polymer data binding
  private errorMessageEnum = ErrorMessage;

  static get is() {
    return 'c2c-error-message';
  }

  static get template() {
    return html`{__html_template__}`;
  }

  private messageCode = ErrorMessage.NO_ERROR;

  setAddSinkError(resultCode: AddSinkResultCode) {
    this.messageCode = this.findErrorMessage(resultCode,
      ErrorMessageElement.ADD_RESULT_MESSAGE_MAP);
  }

  setCastError(resultCode: RouteRequestResultCode) {
    this.messageCode = this.findErrorMessage(resultCode,
      ErrorMessageElement.CAST_RESULT_MESSAGE_MAP);
  }

  setNoError() {
    this.messageCode = ErrorMessage.NO_ERROR;
  }

  getMessageCode() {
    return this.messageCode;
  }

  isEqual(a: ErrorMessage, b: ErrorMessage) {
    return a === b;
  }

  isNotEqual(a: ErrorMessage, b: ErrorMessage) {
    return a !== b;
  }

  private findErrorMessage(
      resultCode: AddSinkResultCode|RouteRequestResultCode,
      messageCodes: Map<ErrorMessage, any[]>) {
    for (const key of messageCodes.keys()) {
      if (messageCodes.get(key)!.includes(resultCode)) {
        return key;
      }
    }

    return ErrorMessage.NO_ERROR;
  }
}

customElements.define(ErrorMessageElement.is, ErrorMessageElement);
