\name{pdb2aln.ind}
\alias{pdb2aln.ind}
\title{ Mapping from alignment positions to PDB atomic indices }
\description{
  Find the best alignment between a PDB structure and an 
  existing alignment. Then, given a set of column indices
  of the original alignment, returns atom selections of 
  equivalent C-alpha atoms in the PDB structure.
}
\usage{
pdb2aln.ind(aln, pdb, inds = NULL, ...)
}
\arguments{
  \item{aln}{ an alignment list object with \code{id} and \code{ali}
    components, similar to that generated by \code{\link{read.fasta}},
    \code{\link{read.fasta.pdb}}, \code{\link{pdbaln}}, and 
    \code{\link{seqaln}}. }
  \item{pdb}{ the PDB object to be aligned to \code{aln}. }
  \item{inds}{ a numeric vector containing a subset of column indices of
    \code{aln}. If NULL, non-gap positions of \code{aln$ali} are used. }
  \item{\dots}{ additional arguments passed to \code{\link{pdb2aln}}. }
}
\details{
  Call \code{\link{pdb2aln}} to align the sequence of \code{pdb} to \code{aln}.
  Then, find the atomic indices of C-alpha atoms in \code{pdb} that are 
  equivalent to \code{inds}, the subset of column indices of \code{aln$ali}. 

  The function is a rountine utility in a combined analysis of 
  molecular dynamics (MD) simulation trajectories and crystallographic 
  structures. For example, a typical post-analysis of MD simulation is to 
  compare the principal components (PCs) derived from simulation trajectories 
  with those derived from crystallographic structures. The C-alpha atoms used
  to fit trajectories and do PCA must be the same (or equivalent) to those
  used in the analysis of crystallographic structures, e.g. the 'non-gap'  
  alignment positions. Call \code{pdb2aln.ind} with providing relevant
  alignment positions, one can easily get equivalent atom selections 
  ('select' class objects) for the simulation topology (PDB) file and then
  do proper trajectory analysis. 
 
}
\value{
  Returns a list containing two "select" objects:
  \item{a}{ atom and xyz indices for the alignment. }
  \item{b}{ atom and xyz indices for the PDB. }

  Note that if any element of \code{inds} has no corresponding CA atom in the 
  PDB, the output \code{a$atom} and \code{b$atom} will be shorter than 
  \code{inds}, i.e. only indices having equivalent CA atoms are returned. 
}
\references{
  Grant, B.J. et al. (2006) \emph{Bioinformatics} \bold{22}, 2695--2696.
}
\author{ Xin-Qiu Yao, Lars Skjaerven & Barry Grant }
\seealso{ \code{\link{seq2aln}}, \code{\link{seqaln.pair}},
       \code{\link{pdb2aln}} }
\examples{
\dontrun{
##--- Read aligned PDB coordinates (CA only)
aln  <- read.fasta(system.file("examples/kif1a.fa",package="bio3d"))
pdbs <- read.fasta.pdb(aln)

##--- Read the topology file of MD simulations
##--- For illustration, here we read another pdb file (all atoms)
pdb <- read.pdb("2kin")

#--- Map the non-gap positions to PDB C-alpha atoms
#pc.inds <- gap.inspect(pdbs$ali)
#npc.inds <- pdb2aln.ind(aln=pdbs, pdb=pdb, inds=pc.inds$f.inds)

#npc.inds$a
#npc.inds$b

#--- Or, map the non-gap positions with a known close sequence in the alignment
#npc.inds <- pdb2aln.ind(aln=pdbs, pdb=pdb, aln.id="1bg2", inds=pc.inds$f.inds)

#--- Map core positions
core <- core.find(pdbs)
core.inds <- pdb2aln.ind(aln=pdbs, pdb=pdb, inds = core$c1A.atom)

core.inds$a
core.inds$b

##--- Fit simulation trajectories to one of the X-ray structures based on
##--- core positions
#xyz <- fit.xyz(pdbs$xyz[1,], pdb$xyz, core.inds$a$xyz, core.inds$b$xyz)

##--- Do PCA of trajectories based on non-gap positions
#pc.traj <- pca(xyz[, npc.inds$b$xyz])

}

}
\keyword{ utilities }

