/*
 * Copyright (c) 2007 QLogic, Inc.  All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 ***************************************************************************
 *
 *   Filename:           qlgc_vnic_dyn_update_daemon.c
 *
 *   Description: 	 This file implemements dynamic update to VNIC module 
 *			 to recover from IOC slot failure/hotswap etc.
 *
 ****************************************************************************/

#include "qlgc_vnic_dyn_update_daemon.h"

#define DAEMON_VERSION "1.0.0.1"
/*
 * Function : allocate_string.
 *
 * Description : Used to allocate all string variables used in daemon.
 */
static char *allocate_string(size_t size)
{
	char *ptr;

	ptr = (char *) malloc (sizeof(char) * size);
	if (!ptr) {
		_DBG_ERROR("Failed to allocate a string.\n");
		exit(1);
	}
	return ptr;
}

/*
 * Function : print_interface_info.
 *
 * Description : prints the interface information on host when debug level > 7.
 */
static void print_interface_info(void)
{
	struct vnic *current;

	for (current = vnic_interfaces; current != NULL ; current=current->next) {
		_DBG_PRINT("Interface Name : %s\n", current->name);
		_DBG_PRINT("RX Checksum : %s\n", current->rx_csum);
		_DBG_PRINT("TX Checksum : %s\n", current->tx_csum);
		_DBG_PRINT("State       : %s\n", state_strings[current->update_state]);
		if (current->primary_path) {
			_DBG_PRINT("Primary Path : \n");
			_DBG_PRINT("\tP Key      : %s\n", current->primary_path->pkey);
			_DBG_PRINT("\tDGID       : %s\n", current->primary_path->dgid);
			_DBG_PRINT("\tIOC GUID   : %s\n", current->primary_path->ioc_guid);
			_DBG_PRINT("\tIOC STRING : %s\n", current->primary_path->ioc_string);
			_DBG_PRINT("\t\tHCA      : %s\n", current->primary_path->hca);
			_DBG_PRINT("\t\tPORT     : %d\n", current->primary_path->port_no);
		}
		if (current->secondary_path) {
			_DBG_PRINT("Secondary Path : \n");
			_DBG_PRINT("\tP Key      : %s\n", current->secondary_path->pkey);
			_DBG_PRINT("\tDGID       : %s\n", current->secondary_path->dgid);
			_DBG_PRINT("\tIOC GUID   : %s\n", current->secondary_path->ioc_guid);
			_DBG_PRINT("\tIOC STRING : %s\n", current->secondary_path->ioc_string);
			_DBG_PRINT("\t\tHCA      : %s\n", current->secondary_path->hca);
			_DBG_PRINT("\t\tPORT     : %d\n", current->secondary_path->port_no);
		}
		_DBG_PRINT("\n");
	}
}

/*
 * Function : init_path_to_default.
 *
 * Description : Initializes path attributes to default values.
 */
static void init_path_to_default(struct netpath *path)
{
	strcpy(path->ioc_guid,   "N/A");
	strcpy(path->ioc_string, "N/A");
	strcpy(path->dgid,       "N/A");
	strcpy(path->pkey,       "N/A");
	strcpy(path->hca,	 "N/A");
	path->port_no = -1;
	path->wait_count = 1;
	path->viport_state = VIPORT_DISCONNECTED;
}

/*
 * Function : allocate_vnic_interface.
 *
 * Description : allocates space to hold information for whole vnic interface.
 */
static struct vnic *allocate_vnic_interface(char *interface_name)
{
	struct vnic *vnic;
	struct netpath *netpath;

	vnic = (struct vnic *) malloc (sizeof(struct vnic));

	strcpy(vnic->name, interface_name);

	netpath = (struct netpath *) malloc (sizeof(struct netpath));
	vnic->primary_path = netpath;
	netpath->vnic = vnic;

	netpath = (struct netpath *) malloc (sizeof(struct netpath));
	vnic->secondary_path = netpath;
	netpath->vnic = vnic;

	vnic->next = NULL;
	vnic->updated = 0;
	vnic->update_state = REQ_NO_UPDATE;

	init_path_to_default(vnic->primary_path);
	init_path_to_default(vnic->secondary_path);

	return vnic;
}

/*
 * Function : free_vnic_interface.
 *
 * Description : frees up the memory allocated for the vnic structure.
 */
static void free_vnic_interface(struct vnic *vnic)
{
	_DBG_INFO("Removing interface %s from daemon interface list.\n",vnic->name);
	free(vnic->primary_path);
	free(vnic->secondary_path);
	free(vnic);
}

/*
 * Function : decode_hca_info
 *
 * Description : decodes the string eg. "vnic-mthca0-1" to find out the 
 *               port no and hca no of the interface.
 */
static void decode_hca_info(struct netpath *path, char *buff)
{
	char *token;

	token = strsep(&buff, "-");
	token = strsep(&buff, "-");
	strncpy(path->hca, token, MAX_HCA_INFO_LENGTH); 
	path->hca[MAX_HCA_INFO_LENGTH] = '\0';
	path->port_no = atoi(buff);
}

/*
 * Function : search_existing_interfaces
 *
 * Description : searches the list of existing interfaces for the given name.
 *               If found, returns pointer to that structure, else NULL.
 */
static struct vnic *search_existing_interfaces(char *name)
{
	struct vnic *current;

	if (vnic_interfaces == NULL) 
		return NULL;

	for (current = vnic_interfaces; current != NULL; current = current->next) {
		if ( !strcmp(name, current->name)) {
			_DBG_PRINT("Found existing interface : %s. Updating the "
					"same.\n", name);
			return current;
		}
	}

	return NULL;
}

/*
 * Function : add_vnic_interface_to_list.
 *
 * Description : checks if interface with given name already exists or not. If 
 *               doesn't exist, adds a new interface to the list.
 */
static struct vnic *add_vnic_interface_to_list(char *name)
{
	struct vnic *current, *vnic;

	vnic = search_existing_interfaces(name);
	if (vnic == NULL) {
		vnic = allocate_vnic_interface(name);
		_DBG_INFO("Adding vnic interface %s to daemon interface list.\n", 
								vnic->name);
		if (vnic_interfaces == NULL) 
			vnic_interfaces = vnic;
		else {
			for (current = vnic_interfaces; current->next != NULL; 
							current = current->next)
				;
			current->next = vnic;
		}
	}

	vnic->primary_path->update_available = NO;
	vnic->primary_path->conn = NULL;
	vnic->secondary_path->update_available = NO;
	vnic->secondary_path->conn = NULL;

	return vnic;
}

/*
 * Function : remove_vnic_interface.
 *
 * Description : removes the vnic interface from the vnic interface list and
 *               frees up the memory held by the interface. 
 */
static void remove_vnic_interface(struct vnic *vnic)
{
	struct vnic *current;

	if (vnic_interfaces == vnic) 
		vnic_interfaces = vnic->next;
	else {
		for (current=vnic_interfaces; current->next != vnic; current=current->next) 
				;
		current->next = vnic->next;
	}
	free_vnic_interface(vnic);
}

/*
 * Function : find_removed_vnic_interfaces.
 *
 * Description : finds up the vnic interfaces that has been removed from the
 *               host and removes those interfaces.
 */
static void find_removed_vnic_interfaces()
{
	struct vnic *current;

	for (current = vnic_interfaces; current != NULL; current=current->next) {
		if (current->updated == 1)
			current->updated = 0;
		else 
			remove_vnic_interface(current);
	}
}
/* 
 * Function : string_lower_case.
 *
 * Description : Converts the string to lower case.
 */
static void string_lower_case(char *string) 
{
	int i;
	for (i = 0; i < strlen(string); i++)
		string[i] = tolower(string[i]);
}

/*
 * Function : read_sys_file.
 *
 * Description : reads the file file_name of size 'size' from sysfs file system
 *               and puts the contents of it buffer vnic_dest.
 */
static short int read_sys_file(char *file_name, size_t size, char *vnic_dest)
{
	int i, file_desc;
	size_t actual_read;
	
	file_desc = open(file_name, O_RDONLY);
	if (file_desc < 0) {
		_DBG_WARN("Failed to read file : %s Error : %s\n",file_name,
						strerror(errno));
		vnic_dest = NULL;
		return -1;
	}

	actual_read = read(file_desc, vnic_dest, size);
	if (actual_read == 0 || actual_read == -1) {
		_DBG_WARN("Failed to read file : %s Error : %s\n", file_name,
						strerror(errno));
		vnic_dest = NULL;
		close(file_desc);
		return -1;
	}

	for (i = 0; vnic_dest[i] != '\n' && vnic_dest[i] != '\0'; i++)
		;
	vnic_dest[i] = '\0';

	vnic_dest[size] = '\0';

	close(file_desc);
	return 0;
}

/*
 * Function : read_sys_path_dir.
 *
 * Description : scans through directories primary_path and secondary_path under the
 *               interface directory in sysfs file system to extract path parameters.
 */
static short int read_sys_path_dir(char *dir_name, struct netpath *path)
{
	DIR *parent_dir;
	struct dirent *dir_ptr;
	char *file_name, *viport_buff, *buffer;
	short int ret;

	file_name = allocate_string(MAX_DIR_ENTRY_NAME);

	parent_dir = opendir(dir_name);
	if (!parent_dir) {
		_DBG_INFO("Error opening path directory : %s Error : %s\n", dir_name,
								strerror(errno));
		goto err_out;	
	}

	while ((dir_ptr = readdir(parent_dir)) != NULL) {

		if (!strcmp(dir_ptr->d_name, ".") ||
                    !strcmp(dir_ptr->d_name, ".."))
                        continue;

		strcpy(file_name, dir_name);
		strcat(file_name, "/");
		strcat(file_name, dir_ptr->d_name);

		if (!strcmp(dir_ptr->d_name, SYS_IOC_GUID)) {
			ret = read_sys_file(file_name, MAX_IOC_GUID_LENGTH, path->ioc_guid);
			string_lower_case(path->ioc_guid);
			continue;
		}	
		
		if (!strcmp(dir_ptr->d_name, SYS_IOC_STRING)) {
			ret = read_sys_file(file_name, MAX_IOC_STRING_LENGTH, path->ioc_string);
			continue;
		}
	
		if (!strcmp(dir_ptr->d_name, SYS_DGID)) {
			ret = read_sys_file(file_name, MAX_DGID_LENGTH, path->dgid);
			string_lower_case(path->dgid);
			continue;
		}	

		if (!strcmp(dir_ptr->d_name, SYS_PKEY)) {
			ret = read_sys_file(file_name, MAX_PKEY_LENGTH, path->pkey);
			string_lower_case(path->pkey);
			continue;
		}

		if (!strcmp(dir_ptr->d_name, SYS_HCA_INFO)) {
			buffer = allocate_string(MAX_DIR_ENTRY_NAME+1);
			ret = read_sys_file(file_name, MAX_DIR_ENTRY_NAME, buffer);
			decode_hca_info(path, buffer);
			free(buffer);
			continue;
		}

		if (!strcmp(dir_ptr->d_name, SYS_VIPORT_STATE)) {
			viport_buff = allocate_string(MAX_VIPORT_STATE_LENGTH+1);
			read_sys_file(file_name, MAX_VIPORT_STATE_LENGTH, viport_buff);
			if(!strcmp(viport_buff, "VIPORT_DISCONNECTED"))
				path->viport_state = VIPORT_DISCONNECTED;
			else 
				path->viport_state = VIPORT_CONNECTED;
			free(viport_buff);
			continue;
		}	
	}

	closedir(parent_dir);
	free(file_name);
	return 0;

err_out:
	free(file_name);
	return -1;
}

/*
 * Function : collect_interface_info.
 *
 * Description : collects the information about the interface from the sysfs file
 *               system by reading the parameters from sysfs files & directories.
 */
static int collect_interface_info(char *interface_dir, char *interface_name)
{
	DIR *parent_dir;
	struct dirent *dir_ptr;
	char *file_name, *interface;
	short int ret;
	struct vnic *vnic;

	vnic = add_vnic_interface_to_list(interface_name);

	interface = allocate_string(MAX_DIR_ENTRY_NAME);
	strcpy(interface, interface_dir);
	strcat(interface, "/");
	strcat(interface, interface_name);

	file_name = allocate_string(MAX_DIR_ENTRY_NAME);

	parent_dir = opendir(interface);
	if (!parent_dir) {
		_DBG_ERROR("Error opening interface directory : %s Error : %s\n", 
						interface, strerror(errno));
		remove_vnic_interface(vnic);
		goto err_out;
	}

	while ((dir_ptr = readdir(parent_dir)) != NULL) {

		if (!strcmp(dir_ptr->d_name, ".") ||
                    !strcmp(dir_ptr->d_name, ".."))
                        continue;

		strcpy(file_name, interface);
		strcat(file_name, "/");
		strcat(file_name, dir_ptr->d_name);

		if (!strcmp(dir_ptr->d_name, SYS_RX_CSUM)) {	
			ret = read_sys_file(file_name, MAX_CHECKSUM_LENGTH, vnic->rx_csum);
			string_lower_case(vnic->rx_csum);
			continue;
		}

		if (!strcmp(dir_ptr->d_name, SYS_TX_CSUM)) {					
			ret = read_sys_file(file_name, MAX_CHECKSUM_LENGTH, vnic->tx_csum);
			string_lower_case(vnic->tx_csum);
			continue;
		}

		if (!strcmp(dir_ptr->d_name, SYS_PRIMARY_PATH)) {
			ret = read_sys_path_dir(file_name, vnic->primary_path);
			continue;
		}

		if (!strcmp(dir_ptr->d_name, SYS_SECONDARY_PATH)) {
			ret = read_sys_path_dir(file_name, vnic->secondary_path);
			continue;
		}
	}

	vnic->updated = 1;

	closedir(parent_dir);
	free(interface);
	free(file_name);
	return 0;

err_out:
	free(interface);
	free(file_name);
	return -1;
}

/*
 * Function : update_criticality.
 *
 * Description : This function marks the VNIC interfaces as critical or non-critical
 *               on the basis of connections of paths of the interface. If there
 *               does not exist at least a single connected path for the interface,
 *               then that interface is critical. If there exists at least one
 *               connected path for the interface, the interface is non-critical.
 */
static void update_criticality()
{
        struct vnic *vnic;

        for (vnic = vnic_interfaces; vnic != NULL; vnic = vnic->next) {

                switch (vnic->primary_path->viport_state) {

                        case VIPORT_CONNECTED    :
                                vnic->primary_path->wait_count = 1;
                                if (vnic->secondary_path->port_no == -1) {
                                        vnic->update_state = REQ_NO_UPDATE;
                                        vnic->secondary_path->wait_count = -1; 
                                } else {
                                        if (vnic->secondary_path->viport_state == VIPORT_CONNECTED) {
                                                vnic->update_state = REQ_NO_UPDATE;
                                                vnic->secondary_path->wait_count = 1;
                                        } else
                                                vnic->update_state = REQ_NON_CRIT_UPDATE;
                                }
                                break;

                        case VIPORT_DISCONNECTED :
                                if (vnic->secondary_path->port_no == -1) {
                                        vnic->update_state = REQ_CRIT_UPDATE;
                                        vnic->secondary_path->wait_count = -1;
                                } else {
                                        if (vnic->secondary_path->viport_state == VIPORT_CONNECTED) {
                                                vnic->update_state = REQ_NON_CRIT_UPDATE;
                                                vnic->secondary_path->wait_count = 1;
                                        } else
                                                vnic->update_state = REQ_CRIT_UPDATE;
                                }
                                break;
                }
                _DBG_PRINT("VNIC interface %s has entered %s state.\n", vnic->name,
                                                state_strings[vnic->update_state]);
        }
}

/*
 * Function : scan_existing_connections.
 *
 * Description : scans existing vnic interfaces on host by reading the directory
 *               /sys/class/infiniband_qlgc_vnic/.
 */
int scan_existing_connections()
{
	DIR *parent_dir;
	struct dirent *dir_ptr;
	short int dir_op_status;
	char *dir_name, *interface_dir = NULL;
	struct stat	statbuf;

	interface_dir = allocate_string(MAX_DIR_ENTRY_NAME); 

	dir_name = allocate_string(MAX_DIR_ENTRY_NAME); 

	strcpy(interface_dir, SYS_VNIC_INTERFACE_DIR);

	parent_dir = opendir(SYS_VNIC_INTERFACE_DIR);
	if (!parent_dir) {
		_DBG_ERROR("%s directory does not exist. Check if the "
		       "qlgc_vnic module is loaded or not.\n", 
			SYS_VNIC_INTERFACE_DIR);
		goto out;
	}	

	while ((dir_ptr = readdir(parent_dir)) != NULL) {
		if (!strcmp(dir_ptr->d_name, ".") ||
		    !strcmp(dir_ptr->d_name, ".."))
			continue;

		strcpy(dir_name, interface_dir);
		strcat(dir_name, "/");
		strcat(dir_name, dir_ptr->d_name);		

		if (lstat(dir_name, &statbuf) < 0) {
			_DBG_ERROR("Error while reading %s Error : %s\n",
						dir_name, strerror(errno));
			closedir(parent_dir);
			goto out;
		}

		if (S_ISDIR(statbuf.st_mode) != 0) {
			dir_op_status = collect_interface_info(interface_dir, 
								dir_ptr->d_name);
			if (dir_op_status < 0)
				goto out;
		}
	}

	dir_op_status = closedir(parent_dir);

	find_removed_vnic_interfaces();
	update_criticality();
	print_interface_info();
	free(interface_dir);
	free(dir_name);
	return 0;

out:
	free(interface_dir);
	free(dir_name);
	return -1;
}
/*---------------------------------------------------------------------------------*/

/*
 * Function : allocate_hca_port_node.
 *
 * Description : allocates space to hold information about the port on which we
 *               successfully got the ibvexdm output.
 */
static struct hca_port_list *allocate_hca_port_node()
{
	struct hca_port_list *port;

	port = (struct hca_port_list *) malloc (sizeof(struct hca_port_list));
	if (!port) 
		_DBG_FATAL("Failed to allocate port structure.\n");

	port->next = NULL;

	return port;
}

/*
 * Function : add_port_to_ibvexdm_list.
 *
 * Description : associates the port information with the ibvexdm connection indicating
 *               from which port we got this ibvexdm output.
 */
static void add_port_to_ibvexdm_list(struct ibvexdm_output *conn,
				    struct hca_port_list  *port)
{
	struct hca_port_list *e_port, *save_port;

	port->updated = 1;
	
	if (conn->port_list == NULL) { 
		conn->port_list = port;
	} else {
		for (e_port = conn->port_list; e_port != NULL; e_port = e_port->next) {
			if (!strcmp(e_port->hca, port->hca) &&
			    e_port->port_no == port->port_no ) {
				e_port->updated = 1;
				free(port);
				return;
			}	
			save_port = e_port;
		}
		save_port->next = port;
	}
	_DBG_INFO("Adding hca : %s, port : %d record for ioc_guid = %s\n", 
				port->hca, port->port_no, conn->ioc_guid);
}

/*
 * Function : remove_port_from_ibvexdm_list.
 *
 * Description : removes the port information associated with the ibvexdm connection.
 */
static void remove_port_from_ibvexdm_list(struct ibvexdm_output *conn,
					 struct hca_port_list  *port)
{
	struct hca_port_list *e_port;
	_DBG_INFO("Removing the hca : %s, port : %d record for ioc_guid = %s.\n",
				port->hca, port->port_no, conn->ioc_guid);

	if (conn->port_list == port) 
		conn->port_list = conn->port_list->next;
	else {
		for (e_port = conn->port_list; e_port->next != port;
							 e_port = e_port->next)
			;
		e_port->next = port->next;
	}	
	free(port);
}

/* Function : found_earlier.
 * 
 * Description : On this run of the daemon have we already decoded any port info
 *               through this umad directory.
 */
static short int found_earlier(char *umad_string)
{
	int i;
	struct hca_port_list *port;

	for (i = 0; i < no_of_umad_devices; i++) {
		port = umad_device_list[i];
		if (!strcmp(umad_string, port->umad_device))
			return 1;
	}

	return 0;
}
/*
 * Function : find_umad_dev_for_path
 *
 * Description : finds the umad device corresponding to the given netpath. 
 */
static short int find_umad_dev_for_path(char *umad_string, struct netpath *path,
						    struct hca_port_list *port)
{
	short int ret;
	DIR *parent_dir;
	struct dirent *dir_ptr;
	char *umad_dir_name, *file_name, *port_buf;

	if (found_earlier(umad_string))
		return 0;

	file_name = allocate_string(MAX_DIR_ENTRY_NAME);

	umad_dir_name = allocate_string(MAX_DIR_ENTRY_NAME);
	strcpy(umad_dir_name, IB_UMAD_DIR);
	strcat(umad_dir_name, umad_string);

	parent_dir = opendir(umad_dir_name);	
	if (!parent_dir) {
                _DBG_ERROR("%s directory does not exist. Check if the "
                       "qlgc_vnic module is loaded or not.\n", umad_dir_name);
		free(file_name);
		free(umad_dir_name);
                return 0;
        }       

	while ((dir_ptr = readdir(parent_dir)) != NULL) {
                if (!strcmp(dir_ptr->d_name, ".") ||
                    !strcmp(dir_ptr->d_name, "..")) 
                        continue;

		strcpy(file_name, umad_dir_name);
		strcat(file_name, "/");
		strcat(file_name, dir_ptr->d_name);		

		if (!strcmp(dir_ptr->d_name, SYS_UMAD_IBDEV)) {
			ret = read_sys_file(file_name, MAX_HCA_INFO_LENGTH, port->hca);
			continue;	
		}	
		
		if (!strcmp(dir_ptr->d_name, SYS_UMAD_PORT)) {
			port_buf = allocate_string(MAX_PORT_LENGTH);
			ret = read_sys_file(file_name, MAX_PORT_LENGTH, port_buf);
			port->port_no = atoi(port_buf);
			free(port_buf);
			continue;
		}
	}

	closedir(parent_dir);
	free(umad_dir_name);
	free(file_name);
	if (!strcmp(path->hca, port->hca) && path->port_no == port->port_no)
		return 1;
	else
		return 0;
}

/*
 * Function : allocate_ibvexdm_output_line.
 *
 * Description : allocates space to hold information of one line after executing
 *               the ibvexdm command on given port.
 */
static struct ibvexdm_output *allocate_ibvexdm_output_line()
{
	struct ibvexdm_output *conn;

	conn = (struct ibvexdm_output *) malloc (sizeof(struct ibvexdm_output));
	if (!conn) 
		_DBG_FATAL("Failed to allocate string structure for ib_qlgc_vnic_query output.\n");

	conn->next = NULL;
	conn->updated = 0;
	conn->port_list  = NULL;

	return conn;
}

/*
 * Function : free_ibvexdm_line.
 *
 * Description : frees up the node structure ibvexdm_output for given ibvexdm line.
 */
static void free_ibvexdm_line(struct ibvexdm_output *conn)
{
	struct hca_port_list *port, *save_port;

	for (port = conn->port_list; port != NULL; ) {
		save_port = port;
		port = port->next;
		remove_port_from_ibvexdm_list(conn, save_port);
	}

	free(conn);
}

/*
 * Function : search_existing_ibvexdm_op.
 *
 * Description : checks if the ibvexdm output with given ioc_guid is already present in the
 *               ibvexdm output list or not.
 */
static struct ibvexdm_output *search_existing_ibvexdm_op(char *ioc_guid)
{
	struct ibvexdm_output *conn;

	if (ibvexdm_op == NULL)
		return NULL;

	for (conn = ibvexdm_op; conn != NULL; conn = conn->next) {
		if (!strcmp(conn->ioc_guid, ioc_guid)) {
			_DBG_PRINT("Found existing ib_qlgc_vnic_query entry : %s. "
				  	"Updating the same.\n", ioc_guid);
			return conn;		
		}
	}	

	return NULL;
}

/*
 * Function : add_line_to_ibvexdm_output.
 *
 * Description : adds the line obtained after running ibvexdm to the ibvexdm_op list.
 */
static struct ibvexdm_output *add_line_to_ibvexdm_output(char *ioc_guid)
{
	struct ibvexdm_output *conn, *line;

	conn = search_existing_ibvexdm_op(ioc_guid);
	if (!conn) {
		conn = allocate_ibvexdm_output_line();	

		if (ibvexdm_op == NULL)
			ibvexdm_op = conn;
		else {
			for (line = ibvexdm_op; line->next != NULL; line = line->next)
				;
			line->next = conn;
		}	
		_DBG_INFO("Adding record for ioc_guid = %s\n", ioc_guid);
	}

	return conn;
}

/*
 * Function : remove_ibvexdm_line.
 *
 * Description : removes the entry of given ibvexdm line from the list of ibvexdm
 *               output lines. 
 */
static void remove_ibvexdm_line(struct ibvexdm_output *line)
{
	struct ibvexdm_output *conn;

	_DBG_INFO("Removing the string with ioc_guid %s from daemon list.\n",line->ioc_guid);

	if (line == ibvexdm_op)
		ibvexdm_op = line->next;
	else {
		for (conn = ibvexdm_op; conn->next != line; conn = conn->next)
			;
		conn->next = line->next;
	}
	free_ibvexdm_line(line);
}

/*
 * Function : scan_port_info.
 *
 * Description : finds out the ports from which we didn't get any valid ib_qlgc_vnic_query output.
 */
static void scan_port_info(struct ibvexdm_output *str, char *umad_device)
{
	struct hca_port_list *port;

	for (port = str->port_list; port != NULL; port = port->next) {
		if (!strcmp(port->umad_device, umad_device)) {
			if (port->updated == 0)
				remove_port_from_ibvexdm_list(str, port);
			else
				port->updated = 0;
		}
	}
}

/*
 * Function : remove_unavailable_ibvexdm_paths.
 *
 * Description : finds and removes the ibvexdm paths which were earlier available but
 *               are not available presently.
 */
static void remove_unavailable_ibvexdm_paths(char *umad_device)
{
	struct ibvexdm_output *str, *save_line;

	for (str = ibvexdm_op; str != NULL; ) {
		scan_port_info(str, umad_device);
		if (str->port_list == NULL) {
			save_line = str;
			str = str->next;
			remove_ibvexdm_line(save_line);
		} else
			str = str->next;
	}
}

/*
 * Function : manipulate_num_string.
 *
 * Description : removes leading zeroes from string.
 */
static void manipulate_num_string(char *string)
{
	int i, j;

	for (i = 0; string[i] == '0'; i++)
		;
	for (j = 0; string[i] != '\0'; i++, j++)
		string[j] = string[i];
	string[j] = '\0';	
}

/*
 * Function : parse_output_line.
 *
 * Description : parses the line obtained after running the ibvexdm on given 
 *               umad device.
 */
static struct ibvexdm_output *parse_output_line(char *line, struct hca_port_list *port)
{
	char *token, *param_name, *buffer;
	struct ibvexdm_output *conn;
	struct hca_port_list *actual_port;

	for ( ; !isalpha(*line); line++); /* Eating up initial white spaces. */

	token = strsep(&line, ",");	/* ioc_guid */
	param_name = strsep(&token, "=");

	if (strcmp(param_name, "ioc_guid"))
		return NULL;

	buffer = allocate_string(MAX_IOC_GUID_LENGTH+1);
	buffer[MAX_IOC_GUID_LENGTH] = '\0';
	strncpy(buffer, token, MAX_IOC_GUID_LENGTH);
	manipulate_num_string(buffer);

	conn = add_line_to_ibvexdm_output(buffer);
	strncpy(conn->ioc_guid, buffer, MAX_IOC_GUID_LENGTH);
	conn->ioc_guid[MAX_IOC_GUID_LENGTH] = '\0';
	free(buffer);

	token = strsep(&line, ",");	/* dgid */
	param_name = strsep(&token, "=");

	if (strcmp(param_name, "dgid")) {
		remove_ibvexdm_line(conn);
		return NULL;	
	}

	manipulate_num_string(token);
	strncpy(conn->dgid, token, MAX_DGID_LENGTH);
	conn->dgid[MAX_DGID_LENGTH] = '\0';

	token = strsep(&line, ",");	/* pkey */
	param_name = strsep(&token, "=");

	if (strcmp(param_name, "pkey")) {
		remove_ibvexdm_line(conn);
		return NULL;
	}
	manipulate_num_string(token);
	strncpy(conn->pkey, token, MAX_PKEY_LENGTH);
	conn->pkey[MAX_PKEY_LENGTH] = '\0';

	line++;				/* ioc_string */
	token = strsep(&line, "\"");
	strncpy(conn->ioc_string, token, MAX_IOC_STRING_LENGTH);
	conn->ioc_string[MAX_IOC_STRING_LENGTH] = '\0';
	
	actual_port = allocate_hca_port_node();
	*actual_port = *port;
	add_port_to_ibvexdm_list(conn, actual_port);

	conn->updated = 1;

	return conn;
}

/* 
 * Function : kill_ibvexdm.
 *
 * Description : This function checks if the ibvexdm output obtained till now
 *               is enough useful to connect all the disconnected paths.
 */
static short int kill_ibvexdm(struct hca_port_list *port)
{
	struct vnic *vnic;

	for (vnic = vnic_interfaces; vnic != NULL; vnic = vnic->next) {
		if ((vnic->primary_path->viport_state == VIPORT_DISCONNECTED   &&
		     vnic->primary_path->update_available == NO                   &&
		    !strcmp(vnic->primary_path->hca, port->hca)                &&
		     vnic->primary_path->port_no == port->port_no)   	
		 || (vnic->secondary_path->port_no != -1                       &&
		     vnic->secondary_path->viport_state == VIPORT_DISCONNECTED &&
		     vnic->secondary_path->update_available == NO		       &&
		    !strcmp(vnic->secondary_path->hca, port->hca)              &&
		     vnic->secondary_path->port_no == port->port_no))
			return 0;
	} 
	
	return 1;
}

/*
 * Function : mark_remaining_ibvexdm_lines_valid.
 *
 * Description : This function marks the all older ibvexdm results obtained through
 *               the given port as valid as we are killing the ibvexdm in middle.
 */
static void mark_remaining_ibvexdm_lines_valid(struct hca_port_list *port)
{
	struct ibvexdm_output *conn;
	struct hca_port_list *hca_ports;

	for (conn = ibvexdm_op; conn != NULL; conn = conn->next) {
		for (hca_ports = conn->port_list; hca_ports != NULL; hca_ports = hca_ports->next) {
			if (!strcmp(hca_ports->hca, port->hca) &&
			    hca_ports->port_no == port->port_no) {
				hca_ports->updated = 1;
				conn->updated = 1;
			}
		}	
	} 
}
/*
 * Function : execute_ibvexdm.
 *
 * Description : executes ibvexdm -es command on given umad device and pipes its
 *               output to stdin from which output is read line by line.
 */
static short int execute_ibvexdm(char *command_string, struct hca_port_list *port)
{
	char *line;
	int fd[2];
	struct ibvexdm_output *conn;

	if (pipe(fd) < 0) { /* Creating pipe between daemon and ibvexdm */
		_DBG_ERROR("Could not execute ib_qlgc_vnic_query : pipe error : %s \n", strerror(errno));
		return -1;
	}

	ibvexdm_pid = fork();	/* New process ibvexdm. */
	if (ibvexdm_pid < 0) {
		_DBG_ERROR("Could not execute ib_qlgc_vnic_query : fork error : %s\n", strerror(errno));
		return -1;
	}
	
	if (ibvexdm_pid > 0) {	/* Parent process : daemon */
		_DBG_PRINT("PID of the ib_qlgc_vnic_query process : %d\n", ibvexdm_pid);
		close(fd[FD_WRITE]);
		if (fd[FD_READ] != STDIN_FILENO) {
                        if (dup2(fd[FD_READ], STDIN_FILENO) != STDIN_FILENO) {
                                _DBG_ERROR("Could not execute ib_qlgc_vnic_query : dup2 error in parent(daemon) : %s\n", strerror(errno));
				kill(ibvexdm_pid, SIGKILL);
				waitpid(ibvexdm_pid, NULL, 0);
                                return -1;
                        }
                        close(fd[FD_READ]);
                }

		line = allocate_string(MAX_LINE_LENGTH);
		while (fgets(line, MAX_LINE_LENGTH-1, stdin) != NULL) {
 	               _DBG_PRINT("ib_qlgc_vnic_query o/p : %s",line);
        	        conn = parse_output_line(line, port);
			if (conn && OPTIMIZE_BEHAVIOUR) {
				scan_vnic_interfaces(conn, port);
				if (kill_ibvexdm(port)) {
					mark_remaining_ibvexdm_lines_valid(port);
					kill(ibvexdm_pid, SIGKILL);
					break;
				}
			}
	        }
		close(fd[FD_READ]);
		if (waitpid(ibvexdm_pid, NULL, 0) < 0) {
			_DBG_ERROR("Could not execute ib_qlgc_vnic_query : waitpid error : %s\n", strerror(errno));
			_DBG_ERROR("If large no of zombies, restart the daemon manually.\n");
			return -1;
		}
		free(line);
		ibvexdm_pid = -1;
	} else {		/* Child Process : ibvexdm */
		close(fd[FD_READ]);
		if (fd[FD_WRITE] != STDOUT_FILENO) {
			if (dup2(fd[FD_WRITE], STDOUT_FILENO) != STDOUT_FILENO) {
				_DBG_ERROR("Could not execute ib_qlgc_vnic_query : dup2 error in child(ib_qlgc_vnic_query) : %s\n", strerror(errno));
				exit(1);
			}
			close(fd[FD_WRITE]);
		}
		if (execl(SHELL, SHELL_COMMAND, SHELL_OPTION, command_string, (char *) 0) < 0) {
			_DBG_ERROR("Could not execute ib_qlgc_vnic_query : execl error : %s\n", strerror(errno));
			exit(1);
		}
	}
	return 0;
}

/*
 * Function : print_ibvexdm_strings.
 *
 * Description : prints the information obtained after running ibvexdm through all
 *               of the umad devices.
 */
static void print_ibvexdm_strings()
{
	struct ibvexdm_output *current;
	struct hca_port_list *port;
	int counter = 1;
		
	for (current = ibvexdm_op; current != NULL; current = current->next) {
		_DBG_PRINT("Line No : %d\n",        counter++);
		_DBG_PRINT("\tIOC GUID   : %s\n",   current->ioc_guid);
		_DBG_PRINT("\tIOC STRING : %s\n",   current->ioc_string);
		_DBG_PRINT("\tDGID       : %s\n",   current->dgid);
		_DBG_PRINT("\tPKEY       : %s\n",   current->pkey);	
		_DBG_PRINT("\tAccessible from following outputs of this host: \n");
		for (port = current->port_list; port != NULL; port = port->next)	
		_DBG_PRINT("\tHCA : %s PORT : %d\n",port->hca, port->port_no);
	}
}

/*
 * Function : decode_port_info.
 *
 * Description : scans through directory /dev/infiniband/ to find out the umad device
 *               corresponding to given vnic netpath.
 */
struct hca_port_list *decode_port_info(struct netpath *path)
{
	DIR *parent_dir;
	struct dirent *dir_ptr;
	short int len;
	char *umad_string;
	struct hca_port_list *port;
	
	umad_string = allocate_string(strlen(UMAD_STRING) + 1);
	port = allocate_hca_port_node(); 

	parent_dir = opendir(IB_DEVICE_DIR);
	if (!parent_dir) {
		_DBG_ERROR("%s directory does not exist. Check if the "
		       "infiniband stack is loaded or not.\n", IB_DEVICE_DIR);
		free(umad_string);
		return NULL;
	}	

	while ((dir_ptr = readdir(parent_dir)) != NULL) {
		if (!strcmp(dir_ptr->d_name, ".") ||
	    	!strcmp(dir_ptr->d_name, "..") )
			continue;

		len = strlen(UMAD_STRING);
		strncpy(umad_string, dir_ptr->d_name, len);
		umad_string[len] = '\0';
		if (!strcmp(umad_string, UMAD_STRING)) {
			if (find_umad_dev_for_path(dir_ptr->d_name, path, port)) {
				strcpy(port->umad_device, dir_ptr->d_name);
				break;
			}
		}
	}

	free(umad_string);

	closedir(parent_dir);
	
	return port;
}

/*----------------------------------------------------------------------------------------*/
/*
 * Function : re_establish_connection.
 *
 * Description : tries to re-establish the connection with VEx for which valid dgid has been
 *               obtained by echoing a string consisting of path parameters to given sysfs
 *               path file of the interface.
 */
static int re_establish_connection(struct ibvexdm_output *conn, struct netpath *path,
			 	    char *path_name)
{
	char *file_name, *echo_string;
	int file_desc, wrote, err_flag = 0;
			
	_DBG_ALWAYS("Trying to re-establish connection for %s path of vnic interface %s.\n",
				path_name, path->vnic->name);

	file_name = allocate_string(MAX_DIR_ENTRY_NAME);
	sprintf(file_name, "/sys/class/infiniband_qlgc_vnic/vnic-%s-%d/create_%s", 
				path->hca, path->port_no, path_name);
	_DBG_PRINT("Path file name : %s\n", file_name);
	
	echo_string = allocate_string(MAX_ECHO_STR_LENGTH);
	sprintf(echo_string, "ioc_guid=%s,ioc_string=\"%s\",dgid=%s,"
			     "pkey=%s,name=%s", conn->ioc_guid, conn->ioc_string,
			     conn->dgid, conn->pkey, path->vnic->name);
	_DBG_PRINT("Echo string : %s\n", echo_string);

	file_desc = open(file_name, O_WRONLY);	
	if (file_desc < 0) {
		_DBG_ERROR("Failed to write in file : %s\n",file_name);
		_DBG_INFO("Failed to re-establish connection for %s path of interface %s.\n",
				path_name, path->vnic->name);
		err_flag = 1;
		goto err_out;
	}

	wrote = write(file_desc, echo_string, strlen(echo_string));
	if (wrote == 0 || wrote == -1 || wrote < strlen(echo_string)) {
		_DBG_ERROR("Failed to write into file : %s the string : %s\n",
			file_name, echo_string);
		_DBG_INFO("Failed to re-establish connection for %s path of interface %s.\n",
				path_name, path->vnic->name);
		err_flag = 1;
	}
err_out:	
	close(file_desc);
	free(file_name);
	free(echo_string);
	path->update_available = NO;
	path->conn = NULL;
	if (err_flag == 1)
		return -1;
	else
		return 0;
}

/*
 * Function : scan_vnic_interfaces.
 *
 * Description : scans the list of vnic interfaces for given ibvexdm output line and port.
 *               checks for same ioc_guid or ioc_string, but different dgid with viport
 *               state as disconnected.
 */
void scan_vnic_interfaces(struct ibvexdm_output *conn, struct hca_port_list *port)
{
	struct vnic *vnic;
	int ret;

	_DBG_PRINT("Scanning vnic interfaces for ioc_guid=%s, HCA=%s, port=%d...\n",
					conn->ioc_guid, port->hca, port->port_no);

	for (vnic = vnic_interfaces; vnic != NULL; vnic = vnic->next) {

		if (vnic->primary_path->viport_state == VIPORT_DISCONNECTED &&
		    vnic->primary_path->update_available == NO                 &&
		    !strcmp(vnic->primary_path->hca, port->hca)             &&
		     vnic->primary_path->port_no == port->port_no ) {
			if ((!strcmp(vnic->primary_path->ioc_string, conn->ioc_string)
			   || !strcmp(vnic->primary_path->ioc_guid, conn->ioc_guid)) &&
			      strcmp(vnic->primary_path->dgid, conn->dgid)) {          
				if (OPTIMIZE_BEHAVIOUR) { 
					_DBG_PRINT("Marking primary path of vnic interface %s.\n", vnic->name);
					vnic->primary_path->update_available = YES;
					vnic->primary_path->conn = conn;
				} else {
					ret = re_establish_connection(conn, vnic->primary_path,
									"primary");
				}
			}
		}

		if (vnic->secondary_path->viport_state == VIPORT_DISCONNECTED &&
		    vnic->secondary_path->viport_state == NO                  &&
		    !strcmp(vnic->secondary_path->hca, port->hca)             &&
		     vnic->secondary_path->port_no == port->port_no ) {
			if ((!strcmp(vnic->secondary_path->ioc_string, conn->ioc_string)
			  || !strcmp(vnic->secondary_path->ioc_guid, conn->ioc_guid)) &&
			      strcmp(vnic->secondary_path->dgid, conn->dgid)) {
				if (OPTIMIZE_BEHAVIOUR) {
					_DBG_PRINT("Marking secondary path of vnic interface %s.\n", vnic->name);
					vnic->secondary_path->update_available = YES;
					vnic->secondary_path->conn = conn;
				} else {
					ret = re_establish_connection(conn, vnic->secondary_path,
									"secondary");
				}
			}
		}
	}
}

/*
 * Function : compare_results.
 *
 * Description : compares the output of ibvexdm with the information about the vnic interfaces
 *               on given host to check if there is possibility of any connection re-establishment.
 */
void compare_results()
{
	struct ibvexdm_output *conn;
	struct hca_port_list  *port;

	for (conn = ibvexdm_op; conn != NULL; conn = conn->next) {
		if (conn->port_list != NULL) {
			for (port = conn->port_list; port != NULL; port = port->next)
				scan_vnic_interfaces(conn, port);
		}
	}
}

/*
 * Function : usage.
 *
 * Description : help menu to use the daemon.
 */
static void usage(char *daemon_name)
{
	fprintf(stderr, "Usage: %s [-v <level>] [-d] [-t <interval>] [-f <factor>] [-o] [-V] \n",daemon_name);
	fprintf(stderr, "    -v - verbosity level\n");
	fprintf(stderr, "    -d - daemon (detach from terminal)\n");
	fprintf(stderr, "    -t - interval in seconds for scanning disconnection\n");
	fprintf(stderr, "    -f - factor of interval(-t) that a non-critical disconnected path waits before daemon attempts connection for it.\n");
	fprintf(stderr, "    -o - Run the daemon in optimzed mode.\n");
	fprintf(stderr, "    -V - QLogic VNIC Dynamic Update Daemon Version\n");
	fprintf(stderr, "\nVerbosity Levels:\n");
	fprintf(stderr, "User can set a value from 0 to 7 based on the following are Verbosity level (default is 6)\n");
	fprintf(stderr, "0, LOG_EMERG,    System is unusable (Not used, reserved for system usage)\n");
	fprintf(stderr, "1, LOG_ALERT,    Action must be taken immediately (Not used, reserved for system usage)\n");
	fprintf(stderr, "2, LOG_CRIT,     Critical conditions\n");
	fprintf(stderr, "3, LOG_ERR,      Error conditions \n");
	fprintf(stderr, "4, LOG_WARNING,  Warning conditions\n");                                                    
	fprintf(stderr, "5, LOG_NOTICE,   Normal but significant condition, Not used, reserved for system usage\n");
	fprintf(stderr, "6, LOG_INFO,     Informational\n");         
	fprintf(stderr, "7, LOG_DEBUG,    Debug-level messages\n ");
}

/*
 * Function : vnic_update_daemon_options.
 *
 * Description : scans for the options to start the vnic dynamic update daemon.
 */
static void vnic_update_daemon_options(int argc, char *argv[])
{
	int c, level, period, factor;
	
	verbosity_level		 = VNIC_UPDATE_DEFAULT_VERBOSITY_LEVEL;
	interval       		 = VNIC_UPDATE_DEFAULT_PERIOD;
	non_critical_path_factor = VNIC_NON_CRITICAL_UPDATE_DEFAULT_FACTOR; 
	OPTIMIZE_BEHAVIOUR	 = NO;

	while (-1 != (c = getopt(argc, argv, "Vdov:t:f:"))) { /* No option mandatory. */
		switch(c) {
			
		case 'd' : 
			is_daemon = TRUE;
			break;

		case 'v' : 
			level = -1;
		   	if ((sscanf(optarg, "%d", &level) != 1) || (level < 0)) { 
                       		_DBG_ALWAYS("Invalid verbose value. Defaulting to %d",
							VNIC_UPDATE_DEFAULT_VERBOSITY_LEVEL);
			} else
				verbosity_level = level;
	                break;
		case 't' : 
			period = -1;
			if ((sscanf(optarg, "%d", &period) != 1) || period < 0) {
				_DBG_ERROR("Invalid interval specified. Defaulting to %d",
								VNIC_UPDATE_DEFAULT_PERIOD);
			} else
				interval = period;
			break;
		case 'f' :
			factor = -1;
			if ((sscanf(optarg, "%d", &factor) != 1) || factor < 0) {
				_DBG_ERROR("Invalid factor specified. Defaulting to %d",
						VNIC_NON_CRITICAL_UPDATE_DEFAULT_FACTOR);
			} else {
				if (!factor)
					non_critical_path_factor = 1;
				else
					non_critical_path_factor = factor;
			}
			break;
		case 'o' :
			_DBG_ALWAYS("Starting the daemon in optimized mode.\n"); 
			OPTIMIZE_BEHAVIOUR = YES;
			break;
		case 'V' :
			printf("QLogic VNIC Dynamic Update Daemon. Version : %s\n", DAEMON_VERSION);
			exit(0);
		default  : 
			usage(argv[0]);
		  	exit(1);
			break;
		}
	}

	_DBG_ALWAYS("Verbosity set to level %d\n", verbosity_level);
	_DBG_ALWAYS("Interval set to %d seconds.\n", interval);
	_DBG_ALWAYS("Non-critical path update factor set to %d.\n", non_critical_path_factor);
}

/*
 * Function : vnic_daemon_child_handler.
 *
 * Description : child communicating to parent that it is running fine and parent can
 *               exit now in case of a daemon mode.
 */
static void vnic_daemon_child_handler()
{
	vnic_child_started = TRUE;
}

/* 
 * Function : reconnect
 *
 * Description : This function attempts a re-connection for disconnected VNIC interface
 *               paths.
 */
static void reconnect()
{
	struct vnic *vnic;
	int ret;

	for (vnic = vnic_interfaces; vnic != NULL; vnic = vnic->next) {
		if (vnic->primary_path->viport_state == VIPORT_DISCONNECTED &&
		    vnic->primary_path->update_available == YES) 
			ret = re_establish_connection(vnic->primary_path->conn, 
							vnic->primary_path, "primary");
		if (vnic->secondary_path->viport_state == VIPORT_DISCONNECTED &&
		    vnic->secondary_path->update_available == YES)
			ret = re_establish_connection(vnic->secondary_path->conn,
							vnic->secondary_path, "secondary");
	}
}

/*
 * Function : need_to_run_ibvexdm.
 * 
 * Description : This function makes sure that when daemon is woken up after its sleep
 *               interval we are not running ibvexdm more than once through the same
 *               umad device.
 */
static int need_to_run_ibvexdm(struct netpath *path, char *path_name)
{
	struct hca_port_list *port;
	short int i;

	if (no_of_umad_devices <= 0)
		return 1;

	for (i = 0; i < no_of_umad_devices; i++) {
		port = umad_device_list[i];
		if (!strcmp(path->hca, port->hca) && path->port_no == port->port_no) {
			_DBG_PRINT("Need not run ib_qlgc_vnic_query on device %s for %s path of VNIC interface %s.\n", 
					port->umad_device, path_name, path->vnic->name);
			return 0;
		}
	}
	
	return 1;
}

/*
 * Function : check_path
 * 
 * Description : This function checks if the given path of the vnic interface is disconnected
 *               or not. If disconnected, it runs ibvexdm through the umad device corresponding
 *               to given path of the vnic interface.
 */
static void check_path(struct netpath *path, char *path_name)
{
	struct hca_port_list *port;
	char *command_string;

	path->wait_count = 1;
	if (path->port_no != -1 && path->viport_state == VIPORT_DISCONNECTED &&
            need_to_run_ibvexdm(path, path_name)) {
		port = decode_port_info(path);
		if (port) {
			command_string = allocate_string(MAX_IBVEXDM_CMD_LENGTH);
		        sprintf(command_string, "ib_qlgc_vnic_query -es -d /dev/infiniband/%s", port->umad_device);
	        	_DBG_INFO("Executing ib_qlgc_vnic_query on device %s for %s path of VNIC interface %s.\n", 
					port->umad_device, path_name, path->vnic->name);
	        	_DBG_PRINT("Command string : %s\n", command_string);
			execute_ibvexdm(command_string, port);
			umad_device_list[no_of_umad_devices++] = port;
			free(command_string);
		} else 
			_DBG_INFO("Could not find umad device for hca %s port %d.\n",
						path->hca, path->port_no);
	}
}

/*
 * Function : update_path_wait_count
 *
 * Description : For a VNIC interface in NON CRITICAL UPDATE state(i.e., at least one 
 *               one path of the interface is connected), we wait for "non_critical_path_factor" 
 *               times the "interval" specified before running ibvexdm for the disconnected path.
 */
static void inline update_path_wait_count(struct netpath *path, char *path_name)
{
        if (path->viport_state == VIPORT_DISCONNECTED) {
                if (path->wait_count == non_critical_path_factor) {
                        _DBG_INFO("Wait for %s path of non-critical VNIC interface %s has elapsed.",
                                        path_name, path->vnic->name);
                        path->wait_count = 1;
                        check_path(path, path_name);
                } else
                        path->wait_count++;
        }
}

/*
 * Function : scan_and_update_vnic_interfaces. 
 *
 * Description : scans the existing vnic interfaces on host and the possible connections
 *               through ibvexdm output. Compares them to find out paths of the
 *               interfaces for which the connections can be established.
 */
static void scan_and_update_vnic_interfaces()
{
	struct vnic *vnic;
	int i;
	struct hca_port_list *port;
 
	no_of_umad_devices = 0;

	if (scan_existing_connections() >= 0) { 
		_DBG_PRINT("Successfully scanned existing vnic interfaces.\n");

		for (vnic = vnic_interfaces; vnic != NULL; vnic = vnic->next) {

                        switch(vnic->update_state) {

                                case REQ_CRIT_UPDATE :
					check_path(vnic->primary_path, "primary");
					check_path(vnic->secondary_path, "secondary");
                                        break;

                                case REQ_NON_CRIT_UPDATE :
                                        update_path_wait_count(vnic->primary_path, "primary");
                                        update_path_wait_count(vnic->secondary_path, "secondary");
                                        break;
                        }
                }


		for (i = 0; i < no_of_umad_devices; i++) {
			port = umad_device_list[i];
			remove_unavailable_ibvexdm_paths(port->umad_device);
			_DBG_PRINT("Freed the umad device entry hca - %s port - %d umad - %s\n",
					port->hca, port->port_no, port->umad_device);
			free(port);
		}
		print_ibvexdm_strings();
		if (OPTIMIZE_BEHAVIOUR)
			reconnect();
		else
			compare_results();
	}	
}

/*
 * Function : cleanup_memory
 * 
 * Description : Frees up the memory allocated for all the interfaces as well as 
 *               each IOController.
 */
static void cleanup_memory()
{
	struct vnic *vnic, *saved;
	struct ibvexdm_output *conn, *saved_conn;

	_DBG_PRINT("Cleaning up the memory.......\n");

	if (ibvexdm_pid != -1) {
		_DBG_PRINT("Sending SIGKILL signal to ib_qlgc_vnic_query PID : %d\n", ibvexdm_pid);
		kill(ibvexdm_pid, SIGKILL);
	}

	for (vnic = vnic_interfaces; vnic != NULL; ) {
		saved = vnic;
		vnic = vnic->next;
		free_vnic_interface(saved);		
	}

	for (conn = ibvexdm_op; conn != NULL; ) {
		saved_conn = conn;
		conn = conn->next;
		free_ibvexdm_line(saved_conn);
	}
}

/*
 * Function : vnic_signal_handler.
 *
 * Description : When running as a daemon, on interrupt stop the daemon.
 */
static void vnic_signal_handler()
{
	vnic_stop_daemon = TRUE;
	cleanup_memory();
	_DBG_ALWAYS("Stopping the vnic update daemon.\n");
	exit(0);
}

/*
 * Function : create_pid_file.
 *
 * Description : Actually creates a pid file at location /var/run.
 */
static void create_pid_file()
{
	int fd;
	char pid_buf[20];

	fd = open(DAEMON_PID_FILE, O_WRONLY | O_CREAT, 0644);
        if (fd >= 0) {
                sprintf(pid_buf, "%d\n", (int)getpid());
                write(fd, pid_buf, strlen(pid_buf));
                close(fd);
		_DBG_ALWAYS("Successfully created a pid file for dynamic update daemon.\n");
        } else
                _DBG_ALWAYS("Failed to create a pid file at location %s\n", DAEMON_PID_FILE);
}

/*
 * Function : create_daemon_pid_file.
 *
 * Description : Creating a pid file in /var/run so that there will not be
 *               multiple instances of daemon running at any time. 
 */
static int create_daemon_pid_file()
{
	int fd, size;
	char pid_buf[20];
	pid_t pid;

	fd = open(DAEMON_PID_FILE, O_RDONLY);
	if (fd >= 0) {
		size = read(fd, pid_buf, sizeof(pid_buf));
		close(fd);
		if (size > 0) {
			pid_buf[size] = '\0';
			pid = atoi(pid_buf);
			/* If the previous daemon is not still running write a
				new pid file immediately.	*/
			if (pid && ((pid == getpid()) || (kill(pid, 0) < 0))) {
				unlink(DAEMON_PID_FILE);
				create_pid_file();
				return 0;
			} else {
				_DBG_ALWAYS("There exists an instance of dynamic update daemon "
						"already running.\n");
				return -1;
			}
		}		
	}
	create_pid_file();
	return 0;
}

/*
 * Function : vnic_initialize_daemon_and_run.
 *
 * Description : If daemon mode selected, initializes the daemon and runs it.
 */
static int vnic_initialize_daemon_and_run()
{
	time_t sec = interval;
	pid_t pid;
	int status;

	if (is_daemon) {		
		signal(SIGUSR1, vnic_daemon_child_handler);
		pid = fork();
		switch(pid) {
			case -1 : 
				_DBG_FATAL("Fork Failed : %s\n", strerror(errno));
				return 1;
				break;

			case 0  :
				if (setsid() < 0)
					exit(1);

				if (chdir("/") < 0) 	/* Daemon requirements. */
					exit(1);

				umask(0); 
				_DBG_ALWAYS("New ib_qlgc_vnic_update starting as daemon.\n");
				break; /* Child process. */

			default : 
				_DBG_ALWAYS("PID of the daemon : %ld.\n", pid);
				sleep(5);
				if (vnic_child_started == FALSE || 
						0 != waitpid(pid, &status, WNOHANG)) {
					_DBG_ALWAYS("Child failed to stay running.\n");
					return 1;
				} 
				return 0;	/* Child started, parent exits. */

		}
		/* Making sure that we do not accept any input from stdin and no output to the
		   terminal or error. */
		(void) freopen("/dev/null", "r", stdin);
		if (is_daemon) {
			(void) freopen("/dev/null", "a", stdout);
			(void) freopen("/dev/null", "a", stderr);
		}
	} else {
		_DBG_ALWAYS("New ib_qlgc_vnic_update starting.\n");
	}

	/* Handle some important signals. */
	if (signal(SIGTERM, vnic_signal_handler) == SIG_ERR)
		goto daemon_exit;

	if (signal(SIGHUP,  vnic_signal_handler) == SIG_ERR)
		goto daemon_exit;

	if (signal(SIGINT,  vnic_signal_handler) == SIG_ERR)
		goto daemon_exit;

	if (is_daemon) {	/* Telling parent that we got through. */
		kill(getppid(), SIGUSR1);
		if (create_daemon_pid_file() < 0)
			return 1;
	}

	while (!vnic_stop_daemon) {
		sleep(sec);
		scan_and_update_vnic_interfaces();
	}

	return 0;

daemon_exit:
	_DBG_ALWAYS("Failed to replace the default signal handler with the "
			"signal handler of the daemon.\n");
	return 1;
}

/*
 * Function : main.
 *
 * Description : main routine which parses the input options and runs the
 *               dynamic update daemon.
 */
int main(int argc, char *argv[])
{
	_DBG_ALWAYS("QLGC_VNIC: Running QLogic Corp. Virtual NIC (VNIC)"
				" Dynamic Update Daemon Version %s \n", DAEMON_VERSION);
	vnic_update_daemon_options(argc, argv);
	return vnic_initialize_daemon_and_run();
}
