#ifndef __INCLUDED_CONFGET_PCRE2_H
#define __INCLUDED_CONFGET_PCRE2_H

/*-
 * Copyright (c) 2021  Peter Pentchev
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#define PCRE2_CODE_UNIT_WIDTH	8

#include <pcre2.h>

#define var_pcre	pcre2_code

static void _do_pcre2_compile(PCRE2_SPTR const pattern, const pcre2_code ** const pvar,
		const char * const name)
{
	int err;
	PCRE2_SIZE ofs;
	static PCRE2_UCHAR err_str[256];

	pcre2_code * const res = pcre2_compile(pattern, PCRE2_ZERO_TERMINATED,
	    0, &err, &ofs, NULL);
	if (res == NULL) {
		pcre2_get_error_message(err, err_str, sizeof(err_str));
		errx(1, "Invalid match %s: %s", name, err_str);
	}
	*pvar = res;
}

static bool _do_pcre2_match(const pcre2_code * const var, const char * const value,
		const char * const pattern)
{
	static pcre2_match_data *data;
	static PCRE2_UCHAR err_str[256];

	if (data == NULL)
		data = pcre2_match_data_create(1, NULL);

	int r = pcre2_match(var, (PCRE2_SPTR)value, strlen(value), 0, 0, data, 0);
	if (r == 0) {
		/* Allocate a larger ovector. */
		pcre2_match_data_free(data);
		data = pcre2_match_data_create_from_pattern(var, NULL);
		r = pcre2_match(var, (PCRE2_SPTR)value, strlen(value), 0, 0, data, 0);
		/* Let us hope r != 0 now. */
	}

	if (r > 0)
		return true;
	if (r == PCRE2_ERROR_NOMATCH)
		return false;

	pcre2_get_error_message(r, err_str, sizeof(err_str));
	errx(1, "Could not match '%s' against the '%s' pattern: %s", value, pattern, err_str);
}

#define do_pcre_compile(pattern, pvar, pvar_extra, name)	\
	_do_pcre2_compile((PCRE2_SPTR)(pattern), (pvar), (name))

#define do_pcre_match(var, var_extra, value, pattern)	\
	_do_pcre2_match((var), (value), (pattern))

#define CONFGET_REGEX_IMPL_PCRE2_(major, minor) #major "." #minor
#define CONFGET_REGEX_IMPL_PCRE2(major, minor) CONFGET_REGEX_IMPL_PCRE2_(major, minor)

#endif
