/****************************************************************************
 *    app/main.cpp - This file is part of coala								*
 *																			*
 *    Copyright (C) 2009  Torsten Grote										*
 *																			*
 *    This program is free software; you can redistribute it and/or modify	*
 *    it under the terms of the GNU General Public License as published by	*
 *    the Free Software Foundation; either version 3 of the License, or		*
 *    (at your option) any later version.									*
 *																			*
 *    This program is distributed in the hope that it will be useful,		*
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of		*
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the			*
 *    GNU General Public License for more details.							*
 *																			*
 *    You should have received a copy of the GNU General Public License		*
 *    along with this program; if not, see http://www.gnu.org/licenses		*
 ****************************************************************************/

#include <iostream>
#include <string>
#include <fstream>
#include "options.h"
#include "../lib/c/Compiler.h"
#include "../lib/c_taid/compiler.h"
#include "../lib/main.h"

using namespace std;
using namespace Coala;

// return values
const int S_UNKNOWN       = 0;
const int S_ERROR         = EXIT_FAILURE;
const int S_MEMORY        = 107;

struct CoalaApp {
	Options options;
	string getFileNames(const std::vector<std::string>& fileList);
	int run(int argc, char** argv);
private:
	std::stringstream ss;
	
	std::istream& getStream(const std::vector<std::string>& fileList) {
		if(fileList.empty() || fileList[0] == "stdin") {
        	return std::cin;
	    }
	    else {
			for(std::vector<std::string>::const_iterator i = fileList.begin(); i != fileList.end(); ++i) {
				std::ifstream inFile(i->c_str());
	            if (!inFile.good()) {
	                throw runtime_error(std::string("Could not open file: ") + *i);
	            }
				else {
					ss << inFile.rdbuf();
				}
	        }
			return ss;
	    }
	}
} coala_app;

string CoalaApp::getFileNames(const std::vector<std::string>& fileList) {
	if(fileList.empty() || fileList[0] == "stdin")
		return "STDIN";
	else {
		string result = "";

		for(std::vector<std::string>::const_iterator i = fileList.begin(); i != fileList.end(); ++i) {
			string tmp_str = *i;
			std::string::size_type pos = tmp_str.find_last_of("/");
			if (pos != std::string::npos)
				tmp_str.erase(0, pos + 1); 
			if(i != fileList.begin()) result += " + ";
			result += tmp_str;
		}
		return result;
	}
}

int CoalaApp::run(int argc, char** argv) {
	if (!options.parse(argc, argv, std::cout)) {
        throw std::runtime_error( options.getError() );
    }
    if (!options.getWarning().empty()) {
        cerr << "coala warning: " << options.getWarning() << endl;
    }
	if (options.help || options.version) {
        return EXIT_SUCCESS;
    }
	
	cout << "% coala " << VERSION << " Translating from action language ";
	if(options.language == "c") cout << "C";
	else if(options.language == "b") cout << "B";
	else if(options.language == "m") cout << "M";
	else if(options.language == "c_taid") cout << "C_taid";
	cout << " with " << (options.fake_class_neg ? "simulated" : "built-in") << " classical negation from ";
	cout << getFileNames(options.input) << " \n";
	
	std::istream& in = getStream(options.input);
	
	Coala::Compiler compiler;
	
	compiler.setLanguage(options.language);
	compiler.setInputStream(in);
	compiler.setOutputStream(cout);
	compiler.setFakeClassicalNegation(options.fake_class_neg);
	compiler.setDebug(options.debug);
	compiler.setDirectEncoding(!options.no_direct_enc);
	compiler.setIncremental(options.incremental);
	compiler.setReverseIncremental(options.reverse);
	compiler.setWhereCheck(!options.no_where_check);
	
	try {
		return compiler.compile();
	}
	catch (const std::exception& e) {
        cerr << "\nERROR: " << e.what() << endl;
        return S_ERROR;
    }
    return S_UNKNOWN;	
}

int main(int argc, char **argv) {
	try {
        return coala_app.run(argc, argv);
    }
    catch (const std::bad_alloc&) {
        cerr << "\nERROR: out of memory"  << endl;
        return S_MEMORY;
    }
    catch (const std::exception& e) {
        cerr << "\nERROR: " << e.what() << endl;
        return S_ERROR;
    }
    return S_UNKNOWN;	
}
