/*LICENSE_START*/
/*
 *  Copyright 1995-2011 Washington University School of Medicine
 *
 *  http://brainmap.wustl.edu
 *
 *  This file is part of CARET.
 *
 *  CARET is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  CARET is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with CARET; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */
/*LICENSE_END*/
#include "CiftiFile.h"
#include "CiftiXML.h"
#include <algorithm>
#include "CiftiXMLReader.h"
#include "CiftiByteSwap.h"

//CiftiFile
CiftiFile::CiftiFile(CACHE_LEVEL clevel) throw (CiftiFileException)
{
   init();
   m_clevel = clevel;
}
CiftiFile::CiftiFile(const QString &fileName,CACHE_LEVEL clevel) throw (CiftiFileException) 
{
   init();   
   this->openFile(fileName, clevel);
}

void CiftiFile::init()
{
   this->m_nifti2Header = NULL;
   this->m_clevel = IN_MEMORY; 
   this->m_xml = NULL;
   this->m_matrix = NULL;
   this->m_swapNeeded = false;
   this->m_copyMatrix = false;
}
void CiftiFile::openFile(const QString &fileName) throw (CiftiFileException)
{
   m_inputFile.setFileName(fileName);
   m_inputFile.open(QIODevice::ReadOnly);
   readHeader();
   
   //read XML
   m_swapNeeded = m_nifti2Header->getSwapNeeded();
   char extensions[4];
   m_inputFile.read(extensions,4);
   unsigned int length;
   m_inputFile.read((char *)&length, 4);
   if(m_swapNeeded)CiftiByteSwap::swapBytes(&length,1);
   unsigned int ecode;
   m_inputFile.read((char *)&ecode,4);
   if(m_swapNeeded)CiftiByteSwap::swapBytes(&ecode,1);
   if(ecode != NIFTI_ECODE_CIFTI) throw CiftiFileException("Error reading extensions.  Extension Code is not Cifti.");
   QByteArray bytes = m_inputFile.read(length-8);//we substract 8 since the length includes the ecode and length
   m_xml = new CiftiXML(bytes);
   
   //read Matrix
   readCiftiMatrix();
}

void CiftiFile::openFile(const QString &fileName, CACHE_LEVEL clevel) throw (CiftiFileException)
{
   m_clevel = clevel;
   this->openFile(fileName);
}
void CiftiFile::writeFile(const QString &fileName) const throw (CiftiFileException)
{
   QFile outputFile(fileName);
   outputFile.open(QIODevice::WriteOnly);
   
   //Get XML string and length, which is needed to calculate the vox_offset stored in the Nifti Header
   QByteArray xmlBytes;
   m_xml->writeXML(xmlBytes);
   int length = 8 + xmlBytes.length();
   int ecode = 32;//NIFTI_ECODE_CIFTI
   char bytes[4] = { 0x01, 0x00,0x00, 0x00};
   
   nifti_2_header header;
   m_nifti2Header->getHeaderStruct(header);
   header.vox_offset = 544 + length;
   m_nifti2Header->SetHeaderStuct(header);
   
   
   //write out the file
   m_nifti2Header->writeFile(outputFile);   
   outputFile.write(bytes,4);
   outputFile.write((char *)&length,4);
   outputFile.write((char *)&ecode,4);   
   outputFile.write(xmlBytes);
   m_matrix->writeMatrix(outputFile);
   outputFile.close();
}

CiftiFile::~CiftiFile()
{
   if(m_nifti2Header) delete m_nifti2Header;
}

// Head IO
void CiftiFile::setHeader(const Nifti2Header &header) throw (CiftiFileException)
{
   if(m_nifti2Header) delete m_nifti2Header;
   m_nifti2Header = new Nifti2Header(header);
}

void CiftiFile::readHeader() throw (CiftiFileException)
{
   if(m_nifti2Header != NULL) delete m_nifti2Header;   
   m_nifti2Header = new Nifti2Header(m_inputFile);  
}

Nifti2Header *CiftiFile::getHeader() throw (CiftiFileException)
{
   if(m_nifti2Header == NULL) readHeader();
   return new Nifti2Header(*m_nifti2Header);   
}

void CiftiFile::getHeader(Nifti2Header &header) throw (CiftiFileException)
{
   if(m_nifti2Header == NULL) readHeader();
   header = *m_nifti2Header;
}

// Matrix IO
void CiftiFile::setCiftiMatrix(CiftiMatrix & matrix) throw (CiftiFileException)
{
   if(this->m_matrix) delete this->m_matrix;
   if(this->m_copyMatrix)      
   {
      this->m_matrix = new CiftiMatrix(matrix);
   }
   else
   {
      this->m_matrix = &matrix;
   }
}

CiftiMatrix * CiftiFile::getCiftiMatrix() throw (CiftiFileException)
{
   if(!this->m_matrix) readCiftiMatrix();//TODO check reset extension offset if needed
   if(this->m_copyMatrix)
   {
      return new CiftiMatrix(*m_matrix);
   }
   else
   {
      CiftiMatrix *temp = this->m_matrix;
      this->m_matrix = NULL;
      return temp;
   }      
}

void CiftiFile::readCiftiMatrix() throw (CiftiFileException)
{
   if(m_matrix != NULL) delete m_matrix;
   std::vector <int> dimensions;
   m_nifti2Header->getCiftiDimensions(dimensions);
   m_matrix = new CiftiMatrix(m_inputFile, dimensions,m_clevel);
   m_matrix->setCopyData(m_copyMatrix);
   if(m_swapNeeded) m_matrix->swapByteOrder();
}

// XML IO
void CiftiFile::setCiftiXML(CiftiXML & xml) throw (CiftiFileException)
{
   if(this->m_xml) delete this->m_xml;
   this->m_xml = new CiftiXML(xml);
   
}

CiftiXML * CiftiFile::getCiftiXML() throw (CiftiFileException)
{
   if(!this->m_xml) return NULL;//readCiftiXML();//TODO check reset extension offset if needed
   return new CiftiXML(*m_xml);
   
}

void CiftiFile::getCiftiXML(CiftiXML &xml) throw (CiftiFileException)
{
   if(!this->m_xml) return;//readCiftiXML();//TODO check reset extension offset if needed
   xml = *m_xml;  
}
